<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB; // Import the DB facade

class MpesaController extends Controller
{
    /**
     * Handles the C2B Confirmation URL call from Safaricom.
     * * IMPORTANT: This code is designed to be fully compatible with older PHP versions
     * and uses the DB facade for reliable database interaction within Laravel.
     */
    public function handleConfirmation(Request $request)
    {
        // --- 1. M-PESA API DATA HANDLING ---
        // Get the raw content directly from the request object
        $mpesaResponse = $request->getContent();
        $data = json_decode($mpesaResponse, true);

        // Check for data presence and required keys
        if (!isset($data['Body']['stkCallback']['CallbackMetadata']['Item'])) {
            // Log unexpected data structure
            error_log(date('Y-m-d H:i:s') . " - Unexpected Mpesa data: " . $mpesaResponse . "\n", 3, 'mpesa_log.log');
            
            // FINAL RESPONSE: Always return success to M-Pesa regardless of internal processing result
            return response()->json(['ResultCode' => 0, 'ResultDesc' => 'Accepted']);
        }

        // Extract transaction details from the 'Item' array
        $transData = $data['Body']['stkCallback']['CallbackMetadata']['Item'];

        $transactionDetails = [];
        foreach ($transData as $item) {
            // Ensure the value exists before assigning
            if (isset($item['Value'])) {
                $transactionDetails[$item['Name']] = $item['Value'];
            }
        }

        // Extract required fields using PHP 5.x compatible 'isset() ? : ' syntax
        $transID = isset($transactionDetails['MpesaReceiptNumber']) ? $transactionDetails['MpesaReceiptNumber'] : 'NO_RECEIPT';
        $amount = isset($transactionDetails['Amount']) ? $transactionDetails['Amount'] : 0.00;
        // BillRefNumber is used here to identify the Borrower's ID Number (based on borrowers.city field)
        $borrowerIdNumber = isset($transactionDetails['BillRefNumber']) ? $transactionDetails['BillRefNumber'] : 'NO_REF'; 

        // Extract date and time from the M-Pesa timestamp
        $rawTime = isset($transactionDetails['TransactionDate']) ? $transactionDetails['TransactionDate'] : date('YmdHis');
        $date = date('Y-m-d', strtotime($rawTime));
        $time = date('H:i:s', strtotime($rawTime));
        $year = date('Y', strtotime($rawTime));
        $month = date('m', strtotime($rawTime));

        // Initialize variables
        $borrower_id = 0;
        $savings_id = 0;
        $old_balance = 0.00;
        $SAVINGS_PRODUCT_ID = 1; // Assuming Savings Product ID 1 is the target account
        
        // Use a try-catch block to handle database exceptions gracefully
        try {
            // --- 2. FIND BORROWER AND SAVINGS ACCOUNT ---

            // Sub-Query 1: Find borrower_id using the ID number (stored in borrowers.city)
            $borrowerRow = DB::table('borrowers')
                ->select('id')
                ->where('city', $borrowerIdNumber)
                ->first();

            if (!$borrowerRow) {
                error_log(date('Y-m-d H:i:s') . " - Borrower ID Number not found: " . $borrowerIdNumber . "\n", 3, 'mpesa_log.log');
                return response()->json(['ResultCode' => 0, 'ResultDesc' => 'Accepted']);
            }

            $borrower_id = $borrowerRow->id;

            // Sub-Query 2: Find the required Savings ID and current balance
            $savingsRow = DB::table('savings')
                ->select('id', 'balance')
                ->where('borrower_id', $borrower_id)
                ->where('savings_product_id', $SAVINGS_PRODUCT_ID)
                ->first();

            if (!$savingsRow) {
                error_log(date('Y-m-d H:i:s') . " - Savings Account (Product ID 1) not found for Borrower ID: " . $borrower_id . "\n", 3, 'mpesa_log.log');
                return response()->json(['ResultCode' => 0, 'ResultDesc' => 'Accepted']);
            }

            $savings_id = $savingsRow->id;
            $old_balance = $savingsRow->balance;
            $new_balance = $old_balance + $amount;


            // --- 3. INSERT INTO savings_transactions ---
            $success = DB::table('savings_transactions')->insert([
                'user_id' => 2, // Default user ID for system transactions
                'borrower_id' => $borrower_id,
                'savings_id' => $savings_id,
                'amount' => $amount,
                'type' => 'deposit',
                'system_interest' => 0,
                'date' => $date,
                'time' => $time,
                'year' => $year,
                'month' => $month,
                'notes' => 'Paid Via Mpesa Api',
                'created_at' => now(),
                'updated_at' => now(),
                'branch_id' => 1, // Default branch ID
                'receipt' => $transID,
                'payment_method_id' => 1, // Default payment method ID
                'debit' => 0.0000,
                'credit' => $amount,
                'balance' => $new_balance,
                'reversible' => 1,
                'reversed' => 0,
                'reversal_type' => 'none',
                'reference' => 0,
            ]);

            if ($success) {
                // --- 4. UPDATE SAVINGS ACCOUNT BALANCE ---
                DB::table('savings')
                    ->where('id', $savings_id)
                    ->update(['balance' => $new_balance]);

                // Log successful deposit
                error_log(date('Y-m-d H:i:s') . " - Transaction SUCCESS for Borrower ID $borrower_id, Savings ID $savings_id, Amount $amount, Receipt $transID\n", 3, 'mpesa_log.log');
            } else {
                // Log failed insert
                error_log(date('Y-m-d H:i:s') . " - DB INSERT FAILED: Unknown reason (check database constraints)\n", 3, 'mpesa_log.log');
            }

        } catch (\Exception $e) {
            // Log any unexpected fatal PHP or DB errors
            error_log(date('Y-m-d H:i:s') . " - FATAL ERROR: " . $e->getMessage() . "\n", 3, 'mpesa_log.log');
        }

        // FINAL RESPONSE: M-Pesa requires a specific JSON response for success
        return response()->json(['ResultCode' => 0, 'ResultDesc' => 'Accepted']);
    }
}
